import os
import time
import json
import threading
from datetime import datetime
from concurrent.futures import ThreadPoolExecutor
import warnings

# Flask and web server components
from flask import Flask, request, jsonify, send_from_directory
from flask_cors import CORS
from werkzeug.utils import secure_filename
from dotenv import load_dotenv

# Audio and data processing
import numpy as np
import librosa
import soundfile as sf
import google.generativeai as genai

# --- Configuration and Constants ---
# Load environment variables from .env file
load_dotenv()

app = Flask(__name__)
CORS(app)

# Configure Paths relative to the app location
BASE_DIR = os.path.dirname(os.path.abspath(__file__))
UPLOAD_FOLDER = os.path.join(BASE_DIR, "uploads")
CONTACT_FOLDER = os.path.join(BASE_DIR, "contacts")
STATIC_FOLDER = os.path.join(BASE_DIR, "static")

app.config["UPLOAD_FOLDER"] = UPLOAD_FOLDER
app.config["CONTACT_FOLDER"] = CONTACT_FOLDER
ALLOWED_EXTENSIONS = {"wav", "mp3", "m4a", "mp4"}

# Ensure directories exist
os.makedirs(UPLOAD_FOLDER, exist_ok=True)
os.makedirs(CONTACT_FOLDER, exist_ok=True)

# --- API Keys ---
GEMINI_API_KEY = os.getenv("GEMINI_API_KEY")
if GEMINI_API_KEY:
    genai.configure(api_key=GEMINI_API_KEY)
else:
    print("⚠️ WARNING: GEMINI_API_KEY not found in environment variables.")

# Filter librosa deprecation warnings
warnings.filterwarnings("ignore", category=FutureWarning)

# --- Mosquito Data ---
# Note: Keys updated to underscores to match your lookup logic
MOSQUITO_BI_DATA = {
    "aedes_aegypti": {
        "species_name": "Aedes aegypti (Linnaeus, 1762)",
        "common_name": "Yellow fever mosquito",
        "description": "Known for transmitting dengue fever, Zika virus, and chikungunya.",
        "habitat": "Urban areas, tropical and subtropical regions.",
        "image": "aedes_aegypti.jpg", # Store in static folder
        "diseases_transmitted": ["Dengue fever", "Zika virus", "Chikungunya", "Yellow fever"],
        "prevention_methods": ["Eliminate standing water", "Use insect repellent"],
        "bioacoustic_index": 181.19,
    },
    "aedes_albopictus": {
        "species_name": "Aedes albopictus (Skuse, 1894)",
        "common_name": "Asian tiger mosquito",
        "description": "Vector for dengue fever and chikungunya.",
        "habitat": "Urban and rural areas.",
        "image": "aedes_albopictus.jpg",
        "diseases_transmitted": ["Dengue fever", "Chikungunya"],
        "prevention_methods": ["Remove standing water", "Use mosquito nets"],
        "bioacoustic_index": 139.70,
    },
    "aedes_vitatus": {
        "species_name": "Aedes vitatus (Bigot, 1861)",
        "common_name": "Eastern saltmarsh mosquito",
        "description": "Found in coastal areas and salt marshes.",
        "habitat": "Salt marshes, brackish water.",
        "image": "aedes_vitatus.jpg",
        "diseases_transmitted": ["Eastern equine encephalitis"],
        "prevention_methods": ["Avoid salt marshes", "Use insect repellent"],
        "bioacoustic_index": 214.90,
    },
    "anopheles_culiciformis": {
        "species_name": "Anopheles culiciformis",
        "common_name": "Culiciformis mosquito",
        "description": "Forest species breeding in tree holes.",
        "habitat": "Forests, bamboo groves.",
        "image": "anopheles_culiciformis.jpg",
        "diseases_transmitted": ["Malaria"],
        "prevention_methods": ["Mosquito nets", "Repellents"],
        "bioacoustic_index": 218.66,
    },
    "anopheles_stephensi": {
        "species_name": "Anopheles stephensi",
        "common_name": "Urban malaria mosquito",
        "description": "Primary urban malaria vector in South Asia.",
        "habitat": "Urban water containers, wells.",
        "image": "anopheles_stephensi.jpg",
        "diseases_transmitted": ["Malaria"],
        "prevention_methods": ["Cover water tanks", "Insecticide"],
        "bioacoustic_index": 263.29,
    },
    "culex_gelidus": {
        "species_name": "Culex gelidus",
        "common_name": "Gelidus mosquito",
        "description": "Vector for Japanese encephalitis.",
        "habitat": "Ponds, marshes.",
        "image": "culex_gelidus.jpg",
        "diseases_transmitted": ["Japanese encephalitis"],
        "prevention_methods": ["Vaccination", "Vector control"],
        "bioacoustic_index": 109.16,
    },
    "culex_mimuloides": {
        "species_name": "Culex mimuloides",
        "common_name": "Mimuloides mosquito",
        "description": "Vector for West Nile virus.",
        "habitat": "Urban freshwater.",
        "image": "culex_mimuloi.jpg",
        "diseases_transmitted": ["West Nile virus"],
        "prevention_methods": ["Reduce breeding sites"],
        "bioacoustic_index": 324.45,
    },
    "mansonia_uniformis": {
        "species_name": "Mansonia uniformis",
        "common_name": "Uniformis mosquito",
        "description": "Vector for filariasis.",
        "habitat": "Ponds with aquatic vegetation.",
        "image": "mansonia_uniformis.jpg",
        "diseases_transmitted": ["Filariasis"],
        "prevention_methods": ["Clear aquatic weeds"],
        "bioacoustic_index": 98.59,
    },
}

def allowed_file(filename):
    return "." in filename and filename.rsplit(".", 1)[1].lower() in ALLOWED_EXTENSIONS

def compute_bioacoustic_index(filepath, fmin=200.0, fmax=700.0, n_fft=2048, hop_length=512):
    try:
        y, sr = librosa.load(filepath, sr=None, mono=True)
        if len(y) == 0: raise ValueError("Empty audio file")

        S = librosa.stft(y, n_fft=n_fft, hop_length=hop_length)
        S_mag = np.abs(S)
        S_db = librosa.amplitude_to_db(S_mag, ref=np.max)
        freqs = librosa.fft_frequencies(sr=sr, n_fft=n_fft)

        band_mask = (freqs >= fmin) & (freqs <= fmax)
        if not np.any(band_mask): return 0.0

        S_band = S_db[band_mask, :]
        S_mean = np.mean(S_band, axis=1)
        S_min = np.min(S_mean)
        BI = np.sum(S_mean - S_min)
        return float(BI)
    except Exception as e:
        print(f"Error computing BI: {e}")
        return 0.0

def classify_mosquito_by_bi(bi_value):
    if not isinstance(bi_value, (int, float)) or bi_value <= 0: return []
    
    results = []
    for species_key, data in MOSQUITO_BI_DATA.items():
        reference_bi = data["bioacoustic_index"]
        difference = abs(bi_value - reference_bi)
        probability = np.exp(-(difference ** 2) / (2 * (50 ** 2)))

        if probability > 0.1:
            percentage = 60 + (probability * 35)
            results.append({
                "species": species_key, # Return the key (e.g. aedes_aegypti)
                "display_name": data["species_name"],
                "percentage_match": round(percentage, 1),
                "bioacoustic_index": float(bi_value),
                "reference_bi": float(reference_bi),
            })

    results.sort(key=lambda x: x["percentage_match"], reverse=True)
    return results[:3]

def process_audio_file(file):
    if not file or not allowed_file(file.filename): return None
    try:
        filename = secure_filename(f"{int(time.time() * 1000)}_{file.filename}")
        filepath = os.path.join(app.config["UPLOAD_FOLDER"], filename)
        file.save(filepath)

        bi_value = compute_bioacoustic_index(filepath)
        classification = classify_mosquito_by_bi(bi_value)

        species_details = None
        if classification:
            # Classification now returns the key directly
            species_key = classification[0]["species"]
            species_details = MOSQUITO_BI_DATA.get(species_key)

        return {
            "filename": filename,
            "bioacoustic_index": bi_value,
            "classification": classification,
            "species_details": species_details,
        }
    except Exception as e:
        print(f"Error in process_audio_file: {e}")
        return None

# --- Routes ---

@app.route("/")
def index():
    return "<h1>Mosquito BI Analysis API is Running on cPanel!</h1>"

@app.route("/static/images/<path:filename>")
def serve_image(filename):
    return send_from_directory(app.config["STATIC_FOLDER"], filename)

@app.route("/analyze", methods=["POST"])
def analyze_audio():
    if "file" not in request.files: return jsonify({"error": "No 'file' part"}), 400
    file = request.files["file"]
    if file.filename == '': return jsonify({"error": "No file selected"}), 400
    
    result = process_audio_file(file)
    if not result: return jsonify({"error": "Error processing file"}), 500

    return jsonify({
        "success": True,
        "file": result["filename"],
        "bioacoustic_index": result["bioacoustic_index"],
        "classification": result["classification"],
        "species_info": result.get("species_details")
    })

@app.route("/species-info/<species_name>", methods=["GET"])
def get_species_info(species_name):
    # Ensure input matches dictionary keys (e.g. "aedes aegypti" -> "aedes_aegypti")
    species_key = species_name.lower().replace(" ", "_")
    if species_key in MOSQUITO_BI_DATA:
        return jsonify({"success": True, "species": MOSQUITO_BI_DATA[species_key]})
    return jsonify({"success": False, "error": "Species not found"}), 404

@app.route("/all-species", methods=["GET"])
def get_all_species():
    species_list = []
    for key, data in MOSQUITO_BI_DATA.items():
        species_list.append({
            "key": key,
            "species_name": data["species_name"],
            "image": f"/static/images/{data.get('image', '')}"
        })
    return jsonify({"success": True, "count": len(species_list), "species": species_list})

# Entry point
if __name__ == "__main__":
    app.run(debug=True)